'use client';

import React, { useState, useEffect } from 'react';
import {
  Users,
  Plus,
  Eye,
  Edit,
  Trash2,
  Shield,
  UserCheck,
  UserX,
  ArrowLeft,
  Crown,
  Clock,
  Activity
} from 'lucide-react';
import Link from 'next/link';
import { useRouter, useParams } from 'next/navigation';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import type { AgencyUser, Department } from '@/types';
import { agencyUserApi, departmentApi } from '@/services/api';

export default function DepartmentUsersPage() {
  const router = useRouter();
  const params = useParams();
  const departmentId = params?.id as string;

  const [users, setUsers] = useState<AgencyUser[]>([]);
  const [department, setDepartment] = useState<Department | null>(null);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [roleFilter, setRoleFilter] = useState<string>('all');
  const [statusFilter, setStatusFilter] = useState<string>('all');

  useEffect(() => {
    if (departmentId) {
      fetchDepartment();
      fetchUsers();
    }
  }, [departmentId]);

  const fetchDepartment = async () => {
    try {
      const data = await departmentApi.getDepartment(departmentId);
      setDepartment({
        id: data.id,
        name: data.name,
        description: data.description || '',
        color: data.color,
        parentDepartmentId: data.parent_department_id,
        managerId: data.manager_id,
        isActive: Boolean(data.is_active),
        userCount: data.user_count || 0,
        createdAt: new Date(data.created_at),
        updatedAt: new Date(data.updated_at)
      });
    } catch (error) {
      console.error('Error fetching department:', error);
    }
  };

  const fetchUsers = async () => {
    try {
      setLoading(true);

      // Fetch all users from API
      const data = await agencyUserApi.getAgencyUsers();

      // Filter users for this department
      const departmentUsers = data
        .filter((user: any) => user.department_id === departmentId)
        .map((user: any) => ({
          id: user.id,
          firstName: user.first_name,
          lastName: user.last_name,
          email: user.email,
          phone: user.phone,
          avatar: user.avatar,
          department: {
            id: user.department_id,
            name: user.department_name || '',
            color: user.department_color || 'bg-gray-500',
            description: ''
          },
          departmentId: user.department_id,
          role: user.role,
          permissions: user.permissions ? (typeof user.permissions === 'string' ? JSON.parse(user.permissions) : user.permissions) : [],
          isActive: Boolean(user.is_active),
          isOnline: Boolean(user.is_online),
          lastLoginAt: user.last_login_at ? new Date(user.last_login_at) : null,
          managerId: user.manager_id,
          totalTickets: user.total_tickets || 0,
          totalRevenue: user.total_revenue || 0,
          createdAt: new Date(user.created_at),
          updatedAt: new Date(user.updated_at)
        }));

      setUsers(departmentUsers);
    } catch (error) {
      console.error('Error fetching users:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleUserStatusChange = async (userId: string, isActive: boolean) => {
    try {
      // Update user status via API
      await agencyUserApi.updateAgencyUser(userId, { is_active: isActive });

      // Update local state
      setUsers(prev => prev.map(user =>
        user.id === userId
          ? { ...user, isActive, updatedAt: new Date() }
          : user
      ));

      alert(isActive ? 'Kullanıcı aktifleştirildi' : 'Kullanıcı pasifleştirildi');
    } catch (error) {
      console.error('Error updating user status:', error);
      alert('Kullanıcı durumu güncellenirken bir hata oluştu');
    }
  };

  const handleUserDelete = async (userId: string) => {
    if (!confirm('Bu kullanıcıyı silmek istediğinize emin misiniz?')) {
      return;
    }

    try {
      // Delete user via API (soft delete)
      await agencyUserApi.deleteAgencyUser(userId);

      // Remove from local state
      setUsers(prev => prev.filter(user => user.id !== userId));

      alert('Kullanıcı başarıyla silindi');
    } catch (error: any) {
      console.error('Error deleting user:', error);
      alert('Kullanıcı silinirken bir hata oluştu: ' + (error.response?.data?.error || error.message));
    }
  };

  // Render kullanıcı bilgileri
  const renderUser = (user: AgencyUser) => (
    <div className="flex items-center">
      <div className="flex-shrink-0 h-10 w-10 rounded-full bg-gradient-to-r from-blue-600 to-purple-600 flex items-center justify-center">
        <span className="text-sm font-medium text-white">
          {user.firstName.charAt(0)}{user.lastName.charAt(0)}
        </span>
      </div>
      <div className="ml-3">
        <div className="text-sm font-medium text-gray-900">
          {user.firstName} {user.lastName}
        </div>
        <div className="text-sm text-gray-500">{user.email}</div>
      </div>
    </div>
  );

  // Render iletişim
  const renderContact = (user: AgencyUser) => (
    <div>
      <div className="text-sm font-medium text-gray-900">{user.email}</div>
      <div className="text-xs text-gray-500">{user.phone}</div>
    </div>
  );

  // Render rol
  const renderRole = (user: AgencyUser) => {
    const roleColors: Record<string, string> = {
      system_admin: 'bg-red-100 text-red-800',
      agency_manager: 'bg-purple-100 text-purple-800',
      department_manager: 'bg-blue-100 text-blue-800',
      senior_specialist: 'bg-green-100 text-green-800',
      specialist: 'bg-indigo-100 text-indigo-800',
      assistant: 'bg-gray-100 text-gray-800',
      intern: 'bg-yellow-100 text-yellow-800'
    };

    const roleLabels: Record<string, string> = {
      system_admin: 'Sistem Yöneticisi',
      agency_manager: 'Acente Yöneticisi',
      department_manager: 'Departman Yöneticisi',
      senior_specialist: 'Kıdemli Uzman',
      specialist: 'Uzman',
      assistant: 'Uzman Yardımcısı',
      intern: 'Stajyer'
    };

    return (
      <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${roleColors[user.role] || 'bg-gray-100 text-gray-800'}`}>
        {(user.role === 'system_admin' || user.role === 'agency_manager') && <Crown className="h-3 w-3 mr-1" />}
        {user.role === 'department_manager' && <Shield className="h-3 w-3 mr-1" />}
        {roleLabels[user.role] || user.role}
      </span>
    );
  };

  // Render durum
  const renderStatus = (user: AgencyUser) => (
    <div className="flex flex-col">
      <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
        user.isActive
          ? 'bg-green-100 text-green-800'
          : 'bg-red-100 text-red-800'
      }`}>
        {user.isActive ? 'Aktif' : 'Pasif'}
      </span>
      {user.isOnline && user.isActive && (
        <span className="inline-flex items-center mt-1 text-xs text-green-600">
          <Activity className="h-3 w-3 mr-1" />
          Çevrimiçi
        </span>
      )}
    </div>
  );

  // Render son giriş
  const renderLastLogin = (user: AgencyUser) => (
    <div className="flex items-center text-sm text-gray-500">
      <Clock className="h-4 w-4 mr-1" />
      {user.lastLoginAt ? (
        <span>{user.lastLoginAt.toLocaleDateString('tr-TR')}</span>
      ) : (
        <span className="italic">Hiç giriş yapmadı</span>
      )}
    </div>
  );

  // Tablo sütunları
  const columns: TableColumn<AgencyUser>[] = [
    {
      key: 'user',
      title: 'Kullanıcı',
      render: (_, record) => renderUser(record)
    },
    {
      key: 'contact',
      title: 'İletişim',
      render: (_, record) => renderContact(record)
    },
    {
      key: 'role',
      title: 'Rol',
      render: (_, record) => renderRole(record)
    },
    {
      key: 'status',
      title: 'Durum',
      render: (_, record) => renderStatus(record)
    },
    {
      key: 'lastLogin',
      title: 'Son Giriş',
      render: (_, record) => renderLastLogin(record)
    }
  ];

  // Tablo aksiyonları
  const actions: TableAction<AgencyUser>[] = [
    {
      key: 'view',
      label: 'Görüntüle',
      icon: Eye,
      onClick: (record) => window.location.href = `/users/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'edit',
      label: 'Düzenle',
      icon: Edit,
      onClick: (record) => window.location.href = `/users/${record.id}/edit`,
      variant: 'secondary'
    },
    {
      key: 'activate',
      label: 'Aktifleştir',
      icon: UserCheck,
      onClick: (record) => handleUserStatusChange(record.id, true),
      variant: 'success',
      condition: (record) => !record.isActive
    },
    {
      key: 'deactivate',
      label: 'Pasifleştir',
      icon: UserX,
      onClick: (record) => handleUserStatusChange(record.id, false),
      variant: 'warning',
      condition: (record) => record.isActive
    },
    {
      key: 'delete',
      label: 'Sil',
      icon: Trash2,
      onClick: (record) => handleUserDelete(record.id),
      variant: 'danger',
      condition: (record) => !record.isActive
    }
  ];

  // Filtreler
  const filters = [
    {
      key: 'role',
      label: 'Rol',
      value: roleFilter,
      onChange: setRoleFilter,
      options: [
        { value: 'all', label: 'Tüm Roller' },
        { value: 'system_admin', label: 'Sistem Yöneticisi' },
        { value: 'agency_manager', label: 'Acente Yöneticisi' },
        { value: 'department_manager', label: 'Departman Yöneticisi' },
        { value: 'senior_specialist', label: 'Kıdemli Uzman' },
        { value: 'specialist', label: 'Uzman' },
        { value: 'assistant', label: 'Uzman Yardımcısı' },
        { value: 'intern', label: 'Stajyer' }
      ]
    },
    {
      key: 'status',
      label: 'Durum',
      value: statusFilter,
      onChange: setStatusFilter,
      options: [
        { value: 'all', label: 'Tüm Durumlar' },
        { value: 'active', label: 'Aktif' },
        { value: 'inactive', label: 'Pasif' }
      ]
    }
  ];

  // Filtrelenmiş kullanıcılar
  const filteredUsers = users.filter(user => {
    const matchesSearch =
      user.firstName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.lastName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.email.toLowerCase().includes(searchTerm.toLowerCase());

    const matchesRole = roleFilter === 'all' || user.role === roleFilter;
    const matchesStatus = statusFilter === 'all' ||
      (statusFilter === 'active' && user.isActive) ||
      (statusFilter === 'inactive' && !user.isActive);

    return matchesSearch && matchesRole && matchesStatus;
  });

  // İstatistikler
  const stats = {
    total: users.length,
    active: users.filter(u => u.isActive).length,
    online: users.filter(u => u.isOnline && u.isActive).length,
    managers: users.filter(u => ['admin', 'manager'].includes(u.role)).length
  };

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-4">
          <Link
            href={`/departments/${departmentId}`}
            className="flex items-center text-gray-600 hover:text-gray-900"
          >
            <ArrowLeft className="h-5 w-5 mr-2" />
            Departmana Dön
          </Link>
          <div className="h-6 border-l border-gray-300"></div>
          {department && (
            <div className="flex items-center">
              <div className={`w-4 h-4 rounded-full ${department.color} mr-2`}></div>
              <h1 className="text-2xl font-bold text-gray-900">{department.name} - Kullanıcılar</h1>
            </div>
          )}
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Users className="h-8 w-8 text-blue-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Toplam Kullanıcı</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.total}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <UserCheck className="h-8 w-8 text-green-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Aktif Kullanıcı</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.active}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Activity className="h-8 w-8 text-orange-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Çevrimiçi</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.online}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Shield className="h-8 w-8 text-purple-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Yöneticiler</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.managers}</p>
            </div>
          </div>
        </div>
      </div>

      <DataTable
        title={`${department?.name || 'Departman'} Kullanıcıları`}
        description="Bu departmandaki kullanıcıları görüntüleyin ve yönetin"
        data={filteredUsers}
        columns={columns}
        actions={actions}
        loading={loading}
        searchable={true}
        searchPlaceholder="Kullanıcı ara..."
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        headerActions={
          <Link
            href={`/users/new?department=${departmentId}`}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
          >
            <Plus className="h-4 w-4" />
            Yeni Kullanıcı Ekle
          </Link>
        }
        emptyTitle="Kullanıcı bulunamadı"
        emptyDescription="Bu departmanda henüz kullanıcı bulunmuyor."
        emptyIcon={Users}
      />
    </div>
  );
}
